########################################################################
# svd
#
#    SVD specific interface routines.
########################################################################

#
# Commands for the SVD
#
set svdRESET    "\004"
set svdCLEAR    $svdRESET
set svdSTOP     "\020"
set svdSTART    "\b"
set svdUPLOAD   "\002"
set svdDOWNLOAD " "
set svdAUTOBAUD "\001"
set svdRESPONSE $svdAUTOBAUD
set svdULSTOP   $svdAUTOBAUD

set svdNEXTTRACK  "N"
set svdNEXTSECTOR "n"


set svdConnectHelp \
"Could not connect to the SVD.

Try pressing the RESET button on the SVD.
Then press the \"Check\" button.

If that doesn't work, the serial port cable
may be disconnected or need to be \"reversed\"."

########################################################################
# svdCheckSerial() -	Checks the serial port and generates a message
#                       is necessary.
########################################################################
proc svdCheckSerial {} {
set svdRS232Help \
"A valid serial/RS232 port has not been selected.  Use the File menu to set the RS232 parameters before attempting this operation."

    if { ! [serialConnected] } {
	tk_messageBox -default ok -icon error -title "ERROR"  \
		      -type ok -message $svdRS232Help
	return 0
    }
    return 1
}

########################################################################
# svdConnect() -	Connects to the SVD...or tries to.  Issues a
#                       reset to get it done.  Pops up the appropriate
#                       messages as it progresses...and updates the right
#                       fields.  Errors come up as necessary too.
#
#       Note that the RS2332(fd) is only valid if "connect" is set.
#       Returns 1 upon success, 0 otherwise.
########################################################################
proc svdConnect {} {
    global svdConnectHelp

    if { ! [svdCheckSerial] } {
	return 0
    }

    workingMessagePopUp "Contacting SVD.  Please Wait."

    if { ! [svdLatch] } {
	workingMessageRemove

	# couldn't contact the SVD for some reason
	# pop-up a message describing the problem

	tk_messageBox -default ok -icon error -title "ERROR"  \
		      -type ok -message $svdConnectHelp

	return 0

    }

    workingMessageRemove

    svdUpdateVersion
    svdVersionSet

    return 1
}

set svd_ok "The SVD seems to be responding just peachy."

########################################################################
# svdCheck() -	Do a quick check to see of the SVD is alive. 
#               Provide feedback.
########################################################################
proc svdCheck {feedback} {
    global svd_ok

    if { [svdConnect] } {
	if { $feedback } {
	    tk_messageBox -default ok -icon info -title "Info"  \
		    -type ok -message $svd_ok
	}
	return 1
    }

    return 0
}

########################################################################
# sendSVDFloppy() -	Send the current floppy out to the SVD for
#                       the given disk.  Assumes that we're connected.
########################################################################
proc sendSVDFloppy {disk callback progress} {
    global RS232
    global floppyMemoryUsed
    global svdMemoryUsed
    global svdDOWNLOAD

    # send the load command

    serialWrite $svdDOWNLOAD
    set data [serialRead 0]

    if { $data != " " } {
	return 0
    }

    serialWrite [floppyNumber $disk]
    serialWrite [floppySectors]
    serialWrite [floppyTracks]
    serialWrite [floppySecSize]

    set count 0
    set intrack 0
    set feedback_count [expr [floppySectorsString] + 1]

    while { $count < [floppyDataSize] } {
	serialWrite [floppyData $count 256]
	set count [expr $count + 256]
	set intrack [expr $intrack + 1]
	while { [lindex [fconfigure $RS232(fd) -queue] 1] != 0 } { }

    global floppyMemoryUsed
    global svdMemoryUsed

	set done $count
	set total [floppyDataSize]
	set num $disk

	set internalProgress [expr $done * 100 / $total]

	gaugeValue [.download childsite].totalgauge [expr $internalProgress * $floppyMemoryUsed($disk) / $svdMemoryUsed + $progress * 100 / $svdMemoryUsed] 0
	gaugeValue [.download childsite].taskgauge $internalProgress 0
	update

#	[$callback $disk $count [floppyDataSize]]

	if { $intrack == $feedback_count } {
	    set intrack 0
	    set data [serialRead 0]
	    if { $data != ">" } {
		return 0
	    }
	}
    }

    return 1
}

########################################################################
# svdUploadFloopy() -	Upload a given floppy image from the SVD.
#                       Assumes that we are connected.
#                       The uploaded file is stored in floppyUploadFile,
#                       and is stored in SVD format.
########################################################################
proc svdUploadFloppy {disk tchan} {
    global RS232
    global floppyProtected
    global floppyUploadStopped
    global svdUPLOAD
    global svdULSTOP

    # clear any data in the RS232 input buffers
    serialRead 0

    #
    # Here's the upload protocol:
    #
    #    - send the upload request byte (which is returned too)
    #    - send the disk number to upload
    #        - response should be 4 bytes:  sectors, track, secsize, writeprot, "+"
    #        - response could be "X" meaning that the requested disk isn't loaded
    #    - when ready for a block, send an "N" for "next" block
    #        - response will be 256 bytes, a ">" if moving to the next track, "+"
    #        - response on last block is 256 bytes, ">", "!"
    #    - if a retransmit is necessay, send a "P" for "previous"
    #        - response will be a "<" if moving to the previous track, 256 bytes, "+"
    #        - NOTE that you can't retransmit the last block (bummer)
    #    - to stop the whole thing, send the stop
    #        - response will be "!"

    # send the upload command for the given disk

    serialWrite $svdUPLOAD
    serialWrite [floppyNumber $disk]

    # assume that the SVD is running pretty fast, and that the data is 
    # ready very quickly after the above xmits - may want to put in a little "after"

    set data [serialRead 0]

    # the \002 gets reflected so we should always get at least 1 char back
    # should have either an X or the header data in "data" now, with a trailing "+" if header data

    if { [string length $data] == 0 } {
	tk_messageBox -default ok -icon error -title "ERROR" -type ok -message "Lost contact with SVD."
	return 0
    }

    if { [string length $data] < 6 } {
	tk_messageBox -default ok -icon error -title "ERROR" -type ok -message \
		      [format "Disk #%d has not been loaded." $disk]
	return 0
    }

    # ignore the leading \002

    binary scan [string range $data 1 4] cccc sectors tracks secsize wprot
    set indicator [string range $data 5 5]

    if { ! [string equal $indicator "+"] } {
	tk_messageBox -default ok -icon error -title "ERROR"  \
		      -type ok -message "Firmware problem - bad upload header indicator"
	return 0
    }

    # header for image composed here

    set floppyProtected($disk) $wprot
    puts -nonewline $tchan [format "2.0\n%d\n%d\n%d\n%d\n%d\n" $sectors $tracks $secsize 1 $wprot]

    # put up a pretty label about sectors/tracks on gauge

    [.upload childsite].total configure -text [format "Uploading - %d tracks %d sectors" $tracks $sectors]
    update

    # now we start a block by block upload of the image
    # note that a block CAN hold 2 sectors, or multiple blocks
    #   can be used for a single sector

    # see the calculation of trackblocks in Firmware/serial.asm:PrepareForData()

    set trackblocks [expr $sectors * (($secsize+1)/2)]
    if { [expr $secsize % 2] == 0 } {
	set trackblocks [expr $trackblocks + (($sectors+1)/2)]
    }
    incr trackblocks

    if { [iniGetAttribute "Program" "SectorUpload"] } {
	if { ! [svdUploadSectors $tchan $sectors $tracks $secsize $trackblocks] } {
	    return 0
	}
    } else {
	if { ! [svdUploadTracks $tchan $sectors $tracks $secsize $trackblocks] } {
	    return 0
	}
    }

    return [svdMemoryUsageCalc 1 $tracks $sectors]
}


########################################################################
# svdUploadSectors() -	Upload a sector at a time.  SLOOOOW
########################################################################

proc svdUploadSectors {tchan sectors tracks secsize trackblocks} {
    global svdNEXTSECTOR
    global floppyUploadStopped
    global svdULSTOP

    set maxprogress [expr $tracks * $trackblocks]
    set progress 0

    # we're looking for 256 + 1 checksum in each block

    for { set i 0 } { $i < $tracks } { incr i } {

	for { set j 0 } { $j < $trackblocks } { incr j } {

	    if { $floppyUploadStopped != 0 } {
		serialWrite $svdULSTOP
		serialRead 0
		return 0
	    }

	    serialWrite $svdNEXTSECTOR
	    set data [serialRead 0]

	    incr progress
	    gaugeValue [.upload childsite].totalgauge [expr $progress * 100 / $maxprogress] 0
	    [.upload childsite].total configure -text [format "Uploading - %d/%d tracks %d/%d sect" \
	                                                        [expr $i + 1] $tracks [expr $j + 1] $sectors]
	    update

	    # the first 256 bytes are data
	    # the 257'th is the checksum
	    # the 258'th is either a track move indicator or the end of block indicator
	    #   if 259'th is there, then we have an end-of-block/disk indicator

	    if { ! [svdChecksum $data] } {
		tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Checksum error"
		return 0
	    }

	    puts -nonewline $tchan [string range $data 0 255]

	    set length [string length $data]
	    set indicator [string range $data end end]
	    set trackmove [string range $data 257 257]

	    # note that trackmove is only valid if length is 259

	    if { $length == 258 } {
		# this is the normal length
    	    } elseif { $length == 259 } {
		if { ! [string equal $trackmove ">"] } {
		    tk_messageBox -default ok -icon error -title "ERROR"  \
			    -type ok -message "Firmware problem - bad track move indicator"
		    return 0
		}
		if { $j != $trackblocks - 1 } {
		    tk_messageBox -default ok -icon error -title "ERROR"  \
			    -type ok -message "Firmware problem - early track move indicator"
		    return 0
		}
	    } else {
		tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Firmware problem - short/long return"
		return 0
	    }

	    if { [string equal $indicator "!"] } {
		if { $j != $trackblocks - 1 || $i != $tracks - 1} {
		    tk_messageBox -default ok -icon error -title "ERROR"  \
			    -type ok -message "Firmware problem - early end of data"
		    return 0
		}
	    } elseif { ! [string equal $indicator "+"] } {
		tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Firmware problem - bad sector indicator"
		return 0
	    }
	}
    }

    return 1
}

########################################################################
# svdUploadTracks() -	Upload a track at a time...FASTER...
########################################################################

proc svdUploadTracks {tchan sectors tracks secsize trackblocks} {
    global svdNEXTTRACK
    global floppyUploadStopped
    global svdULSTOP

    set maxprogress [expr $tracks * $trackblocks]
    set progress 0

    # we're looking for 256 + 1 checksum in each block

    for { set i 0 } { $i < $tracks } { incr i } {

	if { $floppyUploadStopped != 0 } {
	    serialWrite $svdULSTOP
	    serialRead 0
	    return 0
	}

	serialWrite $svdNEXTTRACK
	set data [serialRead 0]

	set progress [expr $progress + $trackblocks]
	gaugeValue [.upload childsite].totalgauge [expr $progress * 100 / $maxprogress] 0
	[.upload childsite].total configure -text [format "Uploading - %d/%d tracks %d sect" \
	                                                               [expr $i + 1] $tracks $sectors]
	update

	# the first $trackblocks * 257 is data
	# the next byte IS a track indicator
	# the last byte IS either a end-of-transfer or end-of-disk indicator

	set length [string length $data]
	set indicator [string range $data end end]
	set trackmove [string range $data [expr $length - 2] [expr $length - 2]]

	if { $length < [expr $trackblocks * 257 + 2] } {
	    tk_messageBox -default ok -icon error -title "ERROR"  \
		    -type ok -message \
		    [format "Firmware problem - short/long return %d (%d)" $length [expr $trackblocks * 257 +2]]
	    return 0
	}

	if { ! [string equal $trackmove ">"] } {
	    tk_messageBox -default ok -icon error -title "ERROR"  \
		    -type ok -message "Firmware problem - bad track move indicator"
	    return 0
	}

	if { [string equal $indicator "!"] } {
	    if { $i != $tracks - 1 } {
		tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Firmware problem - early end of data"
		return 0
	    }
	} elseif { ! [string equal $indicator "+"] } {
	    tk_messageBox -default ok -icon error -title "ERROR"  \
		    -type ok -message "Firmware problem - bad sector indicator"
	    return 0
	}

	for { set j 0 } { $j < [expr $trackblocks * 257]} {set j [expr $j + 257]} {

	    if { ! [svdChecksum [string range $data $j end]] } {
		tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Checksum error"
		return 0
	    }

	    puts -nonewline $tchan [string range $data $j [expr $j + 255]]
	}
    }

    return 1
}


########################################################################
# svdClear() -	Clears the SVD so that a "download all" will work well.
#               In the future, it would be nice if a download of a single
#               disk was possible.  Oh well...
########################################################################
proc svdClear {} {
    global svdCLEAR
    serialWrite $svdCLEAR
}

########################################################################
# svdStop() -	Stop SVD floppy activity...ensures that we don't get
#               completely whacked as accessing memory during floppy
#               operation.      ... stop command is 0x10, 020, ^P
########################################################################
proc svdStop {} {
    global svdSTOP
    serialWrite $svdSTOP
}

########################################################################
# svdStart() -	Start SVD floppy activity
########################################################################
proc svdStart {} {
    global svdSTART
    serialWrite $svdSTART
}


########################################################################
# getSVDversion() -	Get the SVD version number.  Assumes that we're
#                       in operational mode - right baud rate.
########################################################################
proc getSVDversion {} {
    global svdRESPONSE

    # clear any data in the RS232 input buffers
    serialRead 0

    # generate the version message
    serialWrite $svdRESPONSE

    return [serialRead 0]
}

########################################################################
# svdReset() -	Reset the SVD.  Autobaud occurs at the next connect.
########################################################################
proc svdReset {} {

    if { ! [svdCheckSerial] } {
	return 0
    }

    set answer [tk_messageBox -default no -icon warning -title "WARNING"  \
	    -type yesno -parent . -message \
"Are you sure you want to Reset the SVD?
(All previously downloaded images, including
those that have been written to, will be
cleared during reset.)"]

    if { $answer != "yes" } {
	    return 0
    }

    workingMessagePopUp "Reseting the SVD.  Please Wait."

    # first, flush input, then reset
    serialRead 0

    serialBreak

    workingMessageRemove

    return 1
}

########################################################################
# svdLatch() -	Tries to "latch on" to the SVD by sending out the 
#               auto-baud/report/dump-stop character "\001".
#
#               Assumes that the serial port has been configured already.
#
#               Returns TRUE if latched, FALSE if not.
#
#               Caller should wrap a "catch" around call, which indicates
#               a serial port problem.
########################################################################
proc svdLatch {} {

    set LatchAttempts 4

    # first "flush" the input data
    catch {serialRead 0}

    for { set i 0 } { $i < $LatchAttempts } { incr i } {
	serialWrite "\001"
	after 500
	catch {serialRead 0} data
	if { [svdParseVersion $data] } {
	    # we're latched at this point
	    return 1
	}
    }
    # couldn't latch
    return 0
}

########################################################################
# svdParseVersion() -	Assuming that the given data string looks like
#                       a version report, set global version variables
#                       and return TRUE.  Otherwise return FALSE.
#
#                       svdMajorVersion and svdMinorVersion are set if
#                       it parsed, but aren't touched otherwise.
########################################################################
proc svdParseVersion {data} {
    global svdMajorVersion
    global svdMinorVersion

    if { [regexp {^SVD ([0-9][0-9]*)\.([0-9][0-9]*)\001$} $data ignoreMatch major minor] } {
	set svdMajorVersion $major
	set svdMinorVersion $minor
	return 1
    } else {
	return 0
    }
}

########################################################################
# svdChecksum() -	Given a data "block" check to see if the checksum
#                       is correct.  Return TRUE if so.
########################################################################
proc svdChecksum {data} {

    set cs 0

    if { [string length $data] < 257 } {
	return 0
    }

    # get the checksum itself - note that data returned from
    # a binary scan will sign-extend and go negative...which is fixed below

    binary scan [string range $data 256 256] c checksum
    set checksum [expr ( $checksum + 0x100 ) % 0x100]

    binary scan [string range $data 0 255] c256 cdata

    foreach c $cdata {

	# first convert any negative numbers

	set c [expr ( $c + 0x100 ) % 0x100]

	# now do the simple checksum of XOR/rotate-left

	set cs [expr $cs ^ $c]
	set topbit [expr $cs / 128]
	set cs [expr ( $cs * 2 + $topbit ) & 0xff]

    }

    if { $cs == $checksum } {
	return 1
    }

    return 0
}
